#!/bin/sh
# Gloria eo in caelo.
#
#
# LICENSE: GNU GPL 2.0.
# monitor(1) microphone monitor wrapper for sh(1).
# Copyright (C) 2024-25, Marc Fege alias 13MDF / DN9MF.
# E-Mail: 13mdf@fege.net
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License version 2
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
# MA 02110-1301, USA.
#
#
# DESCRIPTION:
# Microphone monitor wrapper program for different operating systems.
# Turns on or off the monitor loopback device of an audio input.
#
#
# EXAMPLES:
# 1.: ./monitor.sh on   # turn microphone monitor on.
# 2.: ./monitor.sh off  # turn microphone monitor off.
# 3.: as 1., with latency of 6 ms:
#     ./monitor.sh on - 6
# 4.: as 3., with output of 'alsa_input.usb-...' and latency of 6 ms:
#     ./monitor.sh on alsa_input.usb-... 6


CP='Copyright (C) 2024-25, Marc Fege.'
PN="`basename $0`"
EM='13mdf@fege.net'
VS='1.2'
DT='2025-03-03'
CPQRACB='13MDF'
CPQRAHAM='DN9MF'
NET='https://www.fege.net/software'
MB='[none]'

EA=' ERROR: too many arguments!  Max: 3.'
EO=" ERROR: '$1': unknown option specified!"
ES=' ERROR: System unknown!'
EN=' ERROR: no value stated!'
EW=' ERROR: wrong value stated!'

SYS="`uname -s`"

FuncHelpShort() {
echo "CALLING: $PN on|off [Source] [Latency]
  -h, --help     Prints out this help.
  -v, --version  Version, date, license.

DESCRIPTION:
  Microphone monitor wrapper program for different operating systems.
  Turns on or off the monitor loopback device of an audio input.

EXAMPLES:
  1.: turn microphone monitor  on:
        $PN on
  2.: turn microphone monitor off:
        $PN off
  3.: as 1., with latency of 6 ms:
        $PN on - 6
  4.: as 3., with output of 'alsa_input.usb-...' and latency of 6 ms:
        $PN on alsa_input.usb-... 6"
}

FuncVersion() {
echo "VERSION, AUTHOR, COPYRIGHT AND CONTACT

Version: $VS, date: $DT.
$CP

  This program is provided by the terms
  of the 'GNU GPL' in version 2.
  The original text of the licence could
  be obtained from:
    http://www.gnu.org/licenses/old~
    -licenses/gpl-2.0

  Internet:
    $NET
  Mailbox (V.34):
    $MB
  Contact:
    E-mail   : $EM
    QRA (CB) : $CPQRACB
    QRA (HAM): $CPQRAHAM"
}

FuncES() {
	echo "$ES" >&2 ; FuncHelpShort >&2 ; exit 1
}

FuncEW() {
	echo "$EW" >&2 ; FuncHelpShort >&2 ; exit 1
}


if [ "$#" -gt '3' ] ; then echo "$EA" >&2 ; FuncHelpShort >&2 ; exit 1
else
case $1 in
0|-0|off|--off)
	case $SYS in
	Linux)		pactl unload-module module-loopback
			pactl unload-module module-echo-cancel
			;;
	NetBSD)		mixerctl -w inputs.mic.mute=on inputs.mic=0 \
				inputs.mic.preamp=off ;;
	FreeBSD)	mixer -rec mic ;;
	*)		FuncES ;;
	esac
	;;
1|-1|on|--on)
	case $SYS in
	Linux)
			case $2 in
				''|-)	unset S ;;
				*)	S="source=$2" ;;
			esac
			case $3 in
				''|-)	L='latency_msec=6' ;;
				[1-9]*)	L="latency_msec=$3" ;;
				*)	FuncEW ;;
			esac
			pactl load-module module-loopback $S $L
			pactl load-module module-echo-cancel
			;;
	NetBSD)		mixerctl -w inputs.mic.mute=off inputs.mic=255 \
				inputs.mic.preamp=on ;;
	FreeBSD)	mixer =rec mic ;;
	*)		FuncES ;;
	esac
	;;
-h*|--help*)	FuncHelpShort ;;
-v*|--version*)	FuncVersion ;;
'')	echo "$EN" >&2 ; FuncHelpShort >&2 ; exit 1 ;;
-*|--*)	echo "$EO" >&2 ; FuncHelpShort >&2 ; exit 1 ;;
*)	FuncEW ;;
esac
fi
