/* Gloria eo in caelo.
 *
 * LICENSE: GNU GPL 2.0.
 * srl(1) Serial line speed emulation.
 * Copyright (C) 2024-25, Marc Fege alias 13MDF / DN9MF.
 * E-Mail: 13mdf@fege.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * DESCRIPTION:
 * Reads DATA given either in $1 or from stdin and prints it's values
 * out again, thus every character of each line.  Comparable to cat(1).
 *
 * EXAMPLES:
 * 1.:	./srl			# Reads from stdin and prints out.
 * 2.:	./srl 300		# Reads from stdin and prints out
 *				# with 300 Bd.
 * 3.:	./srl file.txt		# Reads "file.txt" and prints out.
 * 4.:	./srl file.txt 300	# Reads "file.txt" and prints out
 *				# with 300 Bd.
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#if defined(MSDOS) || defined(_WIN16) || defined(_WINDOWS) \
	|| defined(__WINDOWS__)
		#include <fcntl.h>
		#include <io.h>
		#include <time.h>
	#elif _WIN32
		#include <fcntl.h>
		#include <io.h>
		#include <windows.h>		/* Win32 only.	      */
	#elif __OS2__
		#include <fcntl.h>
		#include <io.h>
		#include <os2.h>		/* OS/2 only.	      */
	#else
		#include <unistd.h>		/* POSIX only.	      */
#endif

#define MICSEC		8000000			/* Timing in us per bd*/

#define	CP	"Copyright (C) 2024-25, Marc Fege."
#define	EM	"13mdf@fege.net"
#define	VI	"0.1"
#define	VS	"1.0"
#define	DI	"2024-05-12"
#define	DT	"2025-03-05"
#define	CPQRACB	"13MDF"
#define	CPQRAHM	"DN9MF"
#define	NET	"https://www.fege.net/software"
#define	MB	"[none]"

#define	E_OPT	" %s: ERROR: illegal option \"%c\".\n"
#define	E_OPTH	" %s: ERROR: no other options possible with \"-h\"" \
		"or \"--help\".\n"
#define	E_OPTV	" %s: ERROR: no other options possible with \"-v\"" \
		"or \"--version\".\n"
#define E_ARG		\
	" %s: ERROR: too many arguments (%d).  Max: 2.\n"
	/* " %d ERROR: Numeric value must be the last!\n" */
#define E_TIME_L	\
	" %s: ERROR: too low baud rate specified (%ld <= 0).\n"
#define E_TIME_H	\
	" %s: ERROR: too high rate specified (%ld)!  Max: %d.\n"
#define E_FILE		" %s: ERROR: cannot open file \"%s\".\n"
#define E_STDOUT	" %s: ERROR writing stdout.\n"

#define	HELP	\
"CALLING: %s [option / FILE / Baud rate]\n\
  -h, --help     Prints out this help.\n\
  -v, --version  Version, date, license.\n\
\n\
EXAMPLES:\n\
  %s              -- Reads from stdin and prints out.\n\
  %s 300          -- Reads from stdin and prints out with 300 Bd.\n\
  %s file.txt     -- Reads \"file.txt\" and prints out.\n\
  %s file.txt 300 -- Reads \"file.txt\" and prints out with 300 Bd.\n"

#define	VERSION	\
"VERSION, AUTHOR, COPYRIGHT AND CONTACT\n\
\n\
Version: "VS", date: "DT".\n\
"CP"\n\
\n\
  This program is provided by the terms\n\
  of the \"GNU GPL\" in version 2.\n\
  The original text of the licence could\n\
  be obtained from:\n\
    http://www.gnu.org/licenses/old~\n\
    -licenses/gpl-2.0\n\
\n\
  Internet:\n\
    "NET"\n\
  Mailbox (V.34):\n\
    "MB"\n\
  Contact:\n\
    E-mail   : "EM"\n\
    QRA (CB) : "CPQRACB"\n\
    QRA (HAM): "CPQRAHM"\n"


void FuncCheckTHL(long t, char *pn) {

	if (t <= 0) {
		fprintf(stderr, E_TIME_L, pn, t);
		fprintf(stderr, HELP, pn, pn, pn, pn, pn);
		exit(EXIT_FAILURE);
	} else if (t > MICSEC) {
		fprintf(stderr, E_TIME_H, pn, t, MICSEC);
		fprintf(stderr, HELP, pn, pn, pn, pn, pn);
		exit(EXIT_FAILURE);
	}

}


void FuncSleep(long t) {

	#if defined(MSDOS) || defined(_WIN16) || defined(_WINDOWS) \
		|| defined(__WINDOWS__)
		clock_t curTime = clock();
		while(clock()-curTime < t/1000); /* Wasting 100% CPU. */
	#elif _WIN32
		Sleep(t/1000);			 /* Win32 only.	      */
	#elif __OS2__
		DosSleep(t/1000);		 /* OS/2 only.	      */
	#else
		usleep(t);			 /* Deprecated POSIX. */
	#endif

}


void Filecopy(FILE *ifp, FILE *ofp) {

	unsigned int c;

	/* Omit \r insertion in stdin and stdout for targets below.   */
	/* Note that for fopen() the "rb" mode is still necessary!    */
	#if defined(MSDOS) || defined(_WIN16) || defined(_WINDOWS) \
		|| defined(__WINDOWS__) || defined(_WIN32) \
		|| defined(__OS2__)
		_setmode(_fileno(stdin), _O_BINARY);
		_setmode(_fileno(stdout), _O_BINARY);
	#endif

	while ((c = getc(ifp)) != EOF) {
		putc(c, ofp);
	}

}


void Filecopy_t(FILE *ifp, FILE *ofp, long *t) {

	unsigned int c;

	/* Omit \r insertion in stdin and stdout for targets below.   */
	/* Note that for fopen() the "rb" mode is still necessary!    */
	#if defined(MSDOS) || defined(_WIN16) || defined(_WINDOWS) \
		|| defined(__WINDOWS__) || defined(_WIN32) \
		|| defined(__OS2__)
		_setmode(_fileno(stdin), _O_BINARY);
		_setmode(_fileno(stdout), _O_BINARY);
	#endif

	while ((c = getc(ifp)) != EOF) {
		FuncSleep(*t);
		putc(c, ofp);
		fflush(stdout);
	}

}


int main(int argc, char *argv[]) {

	char *pn = argv[0], b_h = 0, b_v = 0, c = '\0';
	FILE *fp1 = NULL;/*, *fp2 = NULL;
	unsigned long argci = argc-2; */
	long t = 0;

/*
	if (argc == 1) {
		Filecopy(stdin, stdout);
	}
	else {
		while (--argc > 0) {
			if ((fp1 = fopen(*++argv, "rb")) != NULL) {
				if ((fp2 = fopen(argv[argci], "rb")) != NULL) {
					fclose(fp2);
					Filecopy(fp1, stdout);
				} else {
					FuncCheckTHL(atol(argv[argci]), pn);
					t = MICSEC/atol(argv[argci]);
					FuncCheckTHL(t, pn);
					Filecopy_t(fp1, stdout, &t);
				}
				fclose(fp1);
			} else {
				if ((fp2 = fopen(argv[argci], "rb")) != NULL) {
					fclose(fp2);
					fprintf(stderr, E_ARG, pn);
					fprintf(stderr, HELP, pn, pn, pn, pn, pn);
					exit(EXIT_FAILURE);
				} else {
					FuncCheckTHL(atol(*argv), pn);
					t = MICSEC/atol(*argv);
					FuncCheckTHL(t, pn);
					Filecopy_t(stdin, stdout, &t);
				}
			}
		}
	}
*/
	if (argc > 3) {
		fprintf(stderr, E_ARG, pn, argc-1);
		fprintf(stderr, HELP, pn, pn, pn, pn, pn);
		exit(EXIT_FAILURE);
	} else if (argc == 3) {
		FuncCheckTHL(atol(argv[2]), pn);
		t = MICSEC/atol(argv[2]);
		FuncCheckTHL(t, pn);
		fp1 = fopen(argv[argc-2], "rb");
		if (fp1 == NULL) {
			fprintf(stderr, E_FILE, pn, argv[(argc-2)]);
			fprintf(stderr, HELP, pn, pn, pn, pn, pn);
			exit(EXIT_FAILURE);
		} else {
			Filecopy_t(fp1, stdout, &t);
			fclose(fp1);
		}
	} else if (argc == 2) {
		while (--argc > 0 && (*++argv)[0] == '-') {
			if (strcmp(argv[0], "--help" ) == 0 || \
				strcmp(argv[0], "--usage") == 0) {
				b_h = 1;
				break;
			}
			if (strcmp(argv[0], "--version") == 0) {
				b_v = 1;
				break;
			}
			while ((c = *++argv[0])) {
				switch (c) {
				case 'h':
					b_h = 1;
					break;
				case '?':
					b_h = 1;
					break;
				case 'v':
					b_v = 1;
					break;
				case 'V':
					b_v = 1;
					break;
				default:
					fprintf(stderr, E_OPT, pn, c);
					fprintf(stdout, HELP, \
						pn, pn, pn, pn, pn);
					exit(EXIT_FAILURE);
				}
			}
		}
		if (b_h == 1) {
			if (b_v == 1) {
				fprintf(stderr, E_OPTH, pn);
				fprintf(stdout, HELP, \
					pn, pn, pn, pn, pn);
				exit(EXIT_FAILURE);
			} else {
				fprintf(stdout, HELP, \
					pn, pn, pn, pn, pn);
				exit(EXIT_SUCCESS);
			}
		} else if (b_v == 1) {
			if (b_h == 1) {
				fprintf(stderr, E_OPTV, pn);
				fprintf(stdout, HELP, \
						pn, pn, pn, pn, pn);
				exit(EXIT_FAILURE);
			} else {
				fprintf(stdout, VERSION);
				exit(EXIT_SUCCESS);
			}
		} else {
			fp1 = fopen(argv[0], "rb");
			if (fp1 != NULL) {
				Filecopy(fp1, stdout);
				fclose(fp1);
			} else {
				FuncCheckTHL(atol(argv[0]), pn);
				t = MICSEC/atol(argv[0]);
				FuncCheckTHL(t, pn);
				Filecopy_t(stdin, stdout, &t);
			}
		}
	} else {
		Filecopy(stdin, stdout);
	}

	/* In case of error with stdin, report that to user and exit. */
	if (ferror(stdout)) {
		fprintf(stderr, E_STDOUT, pn);
		exit(EXIT_FAILURE);
	}

	return EXIT_SUCCESS;
}
