#!/bin/sh
# Gloria eo in caelo.
#
#
# LICENSE: GNU GPL 2.0.
# bibtex2refer(1) - conversion script from BibTeX to refer(1)
# bibliography format.
# Copyright (C) 2025, Marc Fege alias 13MDF.
# E-Mail: 13mdf@fege.net
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License version 2
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor,
# Boston, MA 02110-1301, USA.
#
#
# EXAMPLES:
# 1.: ./bibtex2refer.sh            # Reads from stdin, writes to stdout.
# 2.: ./bibtex2refer.sh bibl.bib   # Reads 'bibl.bib', writes to stdout.
# 3.: Reads 'bibl.bib', writes to 'bibl.ref':
#     ./bibtex2refer.sh bibl.bib bibl.ref
# 4.: Reads from stdin, writes to 'bibl.ref':
#     ./bibtex2refer.sh - bibl.ref
# 5.: Reads 'bibl.bib', writes to 'bibl.ref' with an intro & UTF8 conv.:
#     ./bibtex2refer.sh -iu bibl.bib bibl.ref


CP='Copyright (C) 2025, Marc Fege.'
PN="`basename $0`"
EM='13mdf@fege.net'
VS='1.8'
DT='2025-05-01'
CPQRACB='13MDF'
CPQRAHAM='DN9MF'
NET='https://www.fege.net/software'
MB='[none]'

EAn=" ERROR: no arguments allowed with '$1'!"
EA2=' ERROR: too many arguments!  Max: 2.'
EA4=' ERROR: too many arguments!  Max: 4.'
EF=" ERROR: '$1': no such file!"
EI=" ERROR: '$1': files in "'$1 and $2 identical!'
EO=" ERROR: '$1': unknown option specified!"


FuncHelpShort() {
echo "CALLING: $PN [-iu] [FileIn] [FileOut]
  -h|--help              Prints out this help.
  -v|--version           Version, date, license.
  -p|--print             Prints a conversion intro message.

  -i [FileIn] [FileOut]  Adds an intro on top of output.
  -u [FileIn] [FileOut]  Converts UTF-8 chars to groff_char(7).

DESCRIPTION:
  Conversion script from BibTeX to refer(1) bibliography format.

EXAMPLES:
  1.: Reads from stdin, writes to stdout:
        $PN
  2.: Reads 'bibl.bib', writes to stdout:
        $PN bibl.bib
  3.: Reads 'bibl.bib', writes to 'bibl.ref':
        $PN bibl.bib bibl.ref
  4.: Reads from stdin, writes to 'bibl.ref':
        $PN - bibl.ref
  5.: Reads 'bibl.bib', writes to 'bibl.ref' with an intro & UTF-8 conv.
        $PN -iu bibl.bib bibl.ref"
}

FuncVersion() {
echo "VERSION, AUTHOR, COPYRIGHT AND CONTACT

Version: $VS, date: $DT.
$CP

  This program is provided by the terms
  of the 'GNU GPL' in version 2.
  The original text of the licence could
  be obtained from:
    http://www.gnu.org/licenses/old~
    -licenses/gpl-2.0

  Internet:
    $NET
  Mailbox (V.34):
    $MB
  Contact:
    E-mail   : $EM
    QRA (CB) : $CPQRACB
    QRA (HAM): $CPQRAHAM"
}

FuncSwCheck() {
PRECONV="`which preconv`"
PRECONV="`echo $?`"
if [ "$PRECONV" -eq '0' ]
	then	PRECONV='preconv' ; PA='-r'
	else	PRECONV='cat' ; unset PA
		echo " CAUTION: 'preconv' is missing!" \
			' Translation of non-ASCII chars omitted!' >&2
fi
}

FuncIntro() {
if [ "$FI" = '1' ] ; then
echo "\
.\\\" refer(1) bibliography database.
.\\\"
.\\\" BIBLIOGRAPHY FIELD FORMATS AND THEIR COUNTERPARTS:
.\\\"
.\\\"       (...)    =     Non-standard BibTeX field.
.\\\"       [...]    =     refer(1)-compatibility BibTeX field.
.\\\"
.\\\" REFER BibTeX         DESCRIPTION
.\\\" %A    author         Author's name.
.\\\" %B    booktitle      Book containing article referenced.
.\\\" %C    address        City (place of publication).
.\\\" %D    year           Date of publication.
.\\\"       month          Month of publication.
.\\\" %E    editor         Editor of book containing article.
.\\\" %F    [footnote]     Footnote number or label (computed).
.\\\" %G    (doi)          Government order number (DOI).
.\\\"       (isbn)         Non-standard for ISBN.
.\\\"       (issn)         Non-standard for ISSN.
.\\\" %H    [header]       Header commentary, printed before ref.
.\\\" %I    publisher      Issuer (publisher), imply %C.
.\\\"       institution    Name of the institution.
.\\\"       organisation   Inst. done that conf./published manual.
.\\\"       school         Name of the university.
.\\\" %J    journal        Journal containing article.
.\\\" %K    @...{%K...     Keywords to help search for references.
.\\\" %L    (shorttitle)   Label field used by -k option of refer.
.\\\" %M    [memorandum]   Bell Labs Memorandum (undefined).
.\\\" %N    number         Number of issue / Number within a volume.
.\\\"       chapter        Number of a chapter in a book.
.\\\"       edition        Edition number of a book.
.\\\" %O                   Other commentary, printed after ref.
.\\\"       howpublished   A publication notice for unusual public./URL.
.\\\"       note           Notes about the reference.
.\\\"       type           type of the technical report or thesis.
.\\\"       (url)          Non-standard for URLs.
.\\\" %P    pages          Page number(s).
.\\\" %Q    [unreversed]   Corporate, Foreign Author; unreversed author.
.\\\" %R    @UNPUBLISHED{%R...  Report, paper, thesis [no.] (unpubl).
.\\\" %S    series         Series title: +book/jour/conf sect. title.
.\\\" %T    title          Title of article/book: mixed case title fmt.
.\\\" %U    [email]        User email address.
.\\\" %V    volume         Volume number: used with %N.
.\\\" %W    [where]        where?/signature/loc. of conf. (curr. undef.)
.\\\" %X    annote         Abstract: filled txt, matching line-by-line.
.\\\"       (abstract)     Alternative field for 'annote'.
.\\\"       (keywords)     Keywords for tagging and filtering.
.\\\" %Y    [toc]          Table of Contents: ignored by refer.
.\\\" %Z    [references]   References: ignored by refer.
.\\\" %l    [language]     Language of document.
.\\\" %$    [price]        Purchase Price: member/other cost.
.\\\" %*    [copyright]    Copyright Notice: accompanies all fields.
.\\\" %^    [parts]        Contained Parts/Containing Doc (crr. undef.).

"
fi
}

FuncUTF82R() {
if [ "$FU" = '1' ] ; then
	FuncSwCheck
	sed	-e "s/`printf '\303\200'`/\\\(\`A/g" \
		-e "s/`printf '\303\240'`/\\\(\`a/g" \
		-e "s/`printf '\303\201'`/\\\('A/g" \
		-e "s/`printf '\303\241'`/\\\('a/g" \
		-e "s/`printf '\303\202'`/\\\(^A/g" \
		-e "s/`printf '\303\242'`/\\\(^a/g" \
		-e "s/`printf '\303\203'`/\\\(~A/g" \
		-e "s/`printf '\303\243'`/\\\(~a/g" \
		-e "s/`printf '\303\204'`/\\\(:A/g" \
		-e "s/`printf '\303\244'`/\\\(:a/g" \
		-e "s/`printf '\303\205'`/\\\(oA/g" \
		-e "s/`printf '\303\245'`/\\\(oa/g" \
		-e "s/`printf '\303\206'`/\\\(AE/g" \
		-e "s/`printf '\303\246'`/\\\(ae/g" \
		-e "s/`printf '\303\210'`/\\\(\`E/g" \
		-e "s/`printf '\303\250'`/\\\(\`e/g" \
		-e "s/`printf '\303\211'`/\\\('E/g" \
		-e "s/`printf '\303\251'`/\\\('e/g" \
		-e "s/`printf '\303\212'`/\\\(^E/g" \
		-e "s/`printf '\303\252'`/\\\(^e/g" \
		-e "s/`printf '\303\213'`/\\\(:E/g" \
		-e "s/`printf '\303\253'`/\\\(:e/g" \
		-e "s/`printf '\303\214'`/\\\(\`I/g" \
		-e "s/`printf '\303\254'`/\\\(\`i/g" \
		-e "s/`printf '\303\215'`/\\\('I/g" \
		-e "s/`printf '\303\255'`/\\\('i/g" \
		-e "s/`printf '\303\216'`/\\\(^I/g" \
		-e "s/`printf '\303\256'`/\\\(^i/g" \
		-e "s/`printf '\303\217'`/\\\(:I/g" \
		-e "s/`printf '\303\257'`/\\\(:i/g" \
		-e "s/`printf '\303\222'`/\\\(\`O/g" \
		-e "s/`printf '\303\262'`/\\\(\`o/g" \
		-e "s/`printf '\303\223'`/\\\('O/g" \
		-e "s/`printf '\303\263'`/\\\('o/g" \
		-e "s/`printf '\303\224'`/\\\(^O/g" \
		-e "s/`printf '\303\264'`/\\\(^o/g" \
		-e "s/`printf '\303\225'`/\\\(~O/g" \
		-e "s/`printf '\303\265'`/\\\(~o/g" \
		-e "s/`printf '\303\226'`/\\\(:O/g" \
		-e "s/`printf '\303\266'`/\\\(:o/g" \
		-e "s/`printf '\303\230'`/\\\(\/O/g" \
		-e "s/`printf '\303\270'`/\\\(\/o/g" \
		-e "s/`printf '\305\222'`/\\\(OE/g" \
		-e "s/`printf '\303\223'`/\\\(oe/g" \
		-e "s/`printf '\303\231'`/\\\(\`U/g" \
		-e "s/`printf '\303\271'`/\\\(\`u/g" \
		-e "s/`printf '\303\232'`/\\\('U/g" \
		-e "s/`printf '\303\272'`/\\\('u/g" \
		-e "s/`printf '\303\233'`/\\\(^U/g" \
		-e "s/`printf '\303\273'`/\\\(^u/g" \
		-e "s/`printf '\303\234'`/\\\(:U/g" \
		-e "s/`printf '\303\274'`/\\\(:u/g" \
		-e "s/`printf '\304\261'`/\\\(.i/g" \
		-e "s/`printf '\310\267'`/\\\(.j/g" \
		-e "s/`printf '\305\201'`/\\\(\/L/g" \
		-e "s/`printf '\305\202'`/\\\(\/l/g" \
		-e "s/`printf '\303\207'`/\\\(,C/g" \
		-e "s/`printf '\303\247'`/\\\(,c/g" \
		-e "s/`printf '\303\221'`/\\\(~N/g" \
		-e "s/`printf '\303\261'`/\\\(~n/g" \
		-e "s/`printf '\303\220'`/\\\(-D/g" \
		-e "s/`printf '\303\260'`/\\\(Sd/g" \
		-e "s/`printf '\303\235'`/\\\('Y/g" \
		-e "s/`printf '\303\275'`/\\\('y/g" \
		-e "s/`printf '\305\270'`/\\\(:Y/g" \
		-e "s/`printf '\303\277'`/\\\(:y/g" \
		-e "s/`printf '\303\236'`/\\\(TP/g" \
		-e "s/`printf '\303\276'`/\\\(Tp/g" \
		-e "s/`printf '\304\206'`/\\\('C/g" \
		-e "s/`printf '\304\207'`/\\\('c/g" \
		-e "s/`printf '\302\271'`/\\\(S1/g" \
		-e "s/`printf '\302\262'`/\\\(S2/g" \
		-e "s/`printf '\302\263'`/\\\(S3/g" \
		-e "s/`printf '\342\210\227'`/\\\(\*\*/g" \
		-e "s/`printf '\342\210\205'`/\\\(es/g" \
		-e "s/`printf '\303\237'`/\\\(ss/g" \
		-e "s/`printf '\341\272\236'`/\\\(ss/g" \
		-e "s/`printf '\305\240'`/\\\(vS/g" \
		-e "s/`printf '\305\241'`/\\\(vs/g" \
		-e "s/`printf '\305\275'`/\\\(vZ/g" \
		-e "s/`printf '\305\246'`/\\\(vz/g" \
		-e "s/`printf '\342\224\202'`/\\\(br/g" \
		-e "s/`printf '\342\200\276'`/\\\(rn/g" \
		-e "s/`printf '\302\242'`/\\\(ct/g" \
		-e "s/`printf '\342\202\254'`/\\\(eu/g" \
		-e "s/`printf '\302\245'`/\\\(Ye/g" \
		-e "s/`printf '\302\243'`/\\\(Po/g" \
		-e "s/`printf '\302\244'`/\\\(Cs/g" \
		-e "s/`printf '\342\200\260'`/\\\(%0/g" \
		-e "s/`printf '\342\200\262'`/\\\(fm/g" \
		-e "s/`printf '\342\200\263'`/\\\(sd/g" \
		-e "s/`printf '\302\265'`/\\\(mc/g" \
		-e "s/`printf '\302\252'`/\\\(Of/g" \
		-e "s/`printf '\302\272'`/\\\(Om/g" \
		-e "s/`printf '\342\200\224'`/\\\(em/g" \
		-e "s/`printf '\342\200\223'`/\\\(en/g" \
		-e "s/`printf '\302\251'`/\\\(co/g" \
		-e "s/`printf '\302\256'`/\\\(rg/g" \
		-e "s/`printf '\342\204\242'`/\\\(tm/g" \
		-e "s/`printf '\302\247'`/\\\(sc/g" \
		-e "s/`printf '\302\241'`/\\\(r!/g" \
		-e "s/`printf '\302\277'`/\\\(r?/g" \
		-e "s/`printf '\302\260'`/\\\(de/g" \
		-e "s/`printf '\342\200\234'`/\\\(lq/g" \
		-e "s/`printf '\342\200\235'`/\\\(rq/g" \
		-e "s/`printf '\342\200\236'`/\\\(Bq/g" \
		-e "s/`printf '\342\200\232'`/\\\(bq/g" \
		-e "s/`printf '\342\200\230'`/\\\(oq/g" \
		-e "s/`printf '\342\200\231'`/\\\(cq/g" \
		-e "s/`printf '\302\253'`/\\\(Fo/g" \
		-e "s/`printf '\302\273'`/\\\(Fc/g" \
		-e "s/`printf '\342\200\271'`/\\\(fo/g" \
		-e "s/`printf '\342\200\272'`/\\\(fc/g" \
		-e "s/`printf '\342\206\220'`/\\\(<-/g" \
		-e "s/`printf '\342\206\221'`/\\\(ua/g" \
		-e "s/`printf '\342\206\222'`/\\\(->/g" \
		-e "s/`printf '\342\206\223'`/\\\(da/g" \
		-e "s/`printf '\342\206\224'`/\\\(<>/g" \
		-e "s/`printf '\342\206\225'`/\\\(va/g" \
		-e "s/`printf '\342\207\220'`/\\\(lA/g" \
		-e "s/`printf '\342\207\221'`/\\\(uA/g" \
		-e "s/`printf '\342\207\222'`/\\\(rA/g" \
		-e "s/`printf '\342\207\223'`/\\\(dA/g" \
		-e "s/`printf '\342\207\224'`/\\\(hA/g" \
		-e "s/`printf '\342\207\225'`/\\\(vA/g" \
		-e "s/`printf '\342\216\257'`/\\\(an/g" \
		-e "s/`printf '\342\227\213'`/\\\(ci/g" \
		-e "s/`printf '\342\200\242'`/\\\(bu/g" \
		-e "s/`printf '\342\200\240'`/\\\(dg/g" \
		-e "s/`printf '\342\200\241'`/\\\(dd/g" \
		-e "s/`printf '\342\227\212'`/\\\(lz/g" \
		-e "s/`printf '\342\226\241'`/\\\(sq/g" \
		-e "s/`printf '\302\266'`/\\\(ps/g" \
		-e "s/`printf '\302\267'`/\\\(pc/g" \
		-e "s/`printf '\342\210\236'`/\\\(if/g" \
		-e "s/`printf '\342\204\265'`/\\\(Ah/g" \
		-e "s/`printf '\342\204\234'`/\\\(Re/g" \
		-e "s/`printf '\306\222'`/\\\(Fn/g" \
		-e "s/`printf '\316\221'`/\\\(\*A/g" \
		-e "s/`printf '\316\222'`/\\\(\*B/g" \
		-e "s/`printf '\316\223'`/\\\(\*G/g" \
		-e "s/`printf '\316\224'`/\\\(\*D/g" \
		-e "s/`printf '\316\225'`/\\\(\*E/g" \
		-e "s/`printf '\316\226'`/\\\(\*Z/g" \
		-e "s/`printf '\316\227'`/\\\(\*Y/g" \
		-e "s/`printf '\316\230'`/\\\(\*H/g" \
		-e "s/`printf '\316\231'`/\\\(\*I/g" \
		-e "s/`printf '\316\232'`/\\\(\*K/g" \
		-e "s/`printf '\316\233'`/\\\(\*L/g" \
		-e "s/`printf '\316\234'`/\\\(\*M/g" \
		-e "s/`printf '\316\235'`/\\\(\*N/g" \
		-e "s/`printf '\316\236'`/\\\(\*C/g" \
		-e "s/`printf '\316\237'`/\\\(\*O/g" \
		-e "s/`printf '\316\240'`/\\\(\*P/g" \
		-e "s/`printf '\316\241'`/\\\(\*R/g" \
		-e "s/`printf '\316\243'`/\\\(\*S/g" \
		-e "s/`printf '\316\244'`/\\\(\*T/g" \
		-e "s/`printf '\316\245'`/\\\(\*U/g" \
		-e "s/`printf '\316\246'`/\\\(\*F/g" \
		-e "s/`printf '\316\247'`/\\\(\*X/g" \
		-e "s/`printf '\316\250'`/\\\(\*Q/g" \
		-e "s/`printf '\316\251'`/\\\(\*W/g" \
		-e "s/`printf '\316\261'`/\\\(\*a/g" \
		-e "s/`printf '\316\262'`/\\\(\*b/g" \
		-e "s/`printf '\316\263'`/\\\(\*g/g" \
		-e "s/`printf '\316\264'`/\\\(\*d/g" \
		-e "s/`printf '\316\265'`/\\\(\*e/g" \
		-e "s/`printf '\316\266'`/\\\(\*z/g" \
		-e "s/`printf '\316\267'`/\\\(\*y/g" \
		-e "s/`printf '\316\270'`/\\\(\*h/g" \
		-e "s/`printf '\316\271'`/\\\(\*i/g" \
		-e "s/`printf '\316\272'`/\\\(\*k/g" \
		-e "s/`printf '\316\273'`/\\\(\*l/g" \
		-e "s/`printf '\316\274'`/\\\(\*m/g" \
		-e "s/`printf '\316\275'`/\\\(\*n/g" \
		-e "s/`printf '\316\276'`/\\\(\*c/g" \
		-e "s/`printf '\316\277'`/\\\(\*o/g" \
		-e "s/`printf '\317\200'`/\\\(\*p/g" \
		-e "s/`printf '\317\201'`/\\\(\*r/g" \
		-e "s/`printf '\317\203'`/\\\(\*s/g" \
		-e "s/`printf '\317\204'`/\\\(\*t/g" \
		-e "s/`printf '\317\205'`/\\\(\*u/g" \
		-e "s/`printf '\317\225'`/\\\(\*f/g" \
		-e "s/`printf '\317\207'`/\\\(\*x/g" \
		-e "s/`printf '\317\210'`/\\\(\*q/g" \
		-e "s/`printf '\317\211'`/\\\(\*w/g" \
		-e "s/`printf '\317\265'`/\\\(+e/g" \
		-e "s/`printf '\317\221'`/\\\(+h/g" \
		-e "s/`printf '\317\226'`/\\\(+p/g" \
		-e "s/`printf '\317\206'`/\\\(+f/g" \
		-e "s/`printf '\317\202'`/\\\(ts/g" \
		-e "s/`printf '\342\231\240'`/\\\(SP/g" \
		-e "s/`printf '\342\231\243'`/\\\(CL/g" \
		-e "s/`printf '\342\231\245'`/\\\(HE/g" \
		-e "s/`printf '\342\231\246'`/\\\(DI/g" \
		-e "s/`printf '\342\200\246'`/.../g" \
		-e "s/`printf '\302\240'`/\\~/g" \
		-e "s/`printf '\342\200\257'`/\\~/g" \
		| $PRECONV $PA
	else	cat
fi
}

FuncChB2R() {
	sed	-e 's/\~/\\\~/g' \
		-e 's/{\\ss}/\\\(ss/g' \
		-e 's/\\"ss/\\\(ss/g' \
		-e 's/\\ss{}/\\\(ss/g' \
		-e 's/{\\SS}/\\\(ss/g' \
		-e 's/\\"SS/\\\(ss/g' \
		-e 's/\\SS{}/\\\(ss/g' \
		-e 's/\\MakeUppercase{\\ss}/\\\(ss/g' \
		-e 's/{\\`\(.\)}/\\\(`\1/g' \
		-e 's/\\`{\(.\)}/\\\(`\1/g' \
		-e 's/{\\'"'"'\(.\)}/\('"'"'\1/g' \
		-e 's/\\'"'"'{\(.\)}/\\\(''\1/g' \
		-e 's/{\\\^\(.\)}/\\\(\^\1/g' \
		-e 's/\\\^{\(.\)}/\\\(\^\1/g' \
		-e 's/\\\^\(.\)/\\\(\^\1/g' \
		-e 's/{\\"\(.\)}/\\\(:\1/g' \
		-e 's/\\"{\(.\)}/\\\(:\1/g' \
		-e 's/\\"\([[:alpha:]]\)/\\\(:\1/g' \
		-e 's/{\\H\(.\)}/\\\(:\1/g' \
		-e 's/\\H{\(.\)}/\\\(:\1/g' \
		-e 's/{\\~\(.\)}/\\\(~\1/g' \
		-e 's/\\~{\(.\)}/\\\(~\1/g' \
		-e 's/{\\c\(.\)}/\\\(,\1/g' \
		-e 's/\\c{\(.\)}/\\\(,\1/g' \
		-e 's/\\l{}/\\\(\/l/g' \
		-e 's/\\L{}/\\\(\/L/g' \
		-e 's/{\\\/\(.\)}/\\\(\/\1/g' \
		-e 's/\\\/{\(.\)}/\\\(\/\1/g' \
		-e 's/{\\r\(.\)}/\\\(o\1/g' \
		-e 's/\\r{\(.\)}/\\\(o\1/g' \
		-e 's/\\aa/\\\(oa/g' \
		-e 's/\\AA/\\\(oA/g' \
		-e 's/{\\v\(.\)}/\\\(v\1/g' \
		-e 's/\\v{\(.\)}/\\\(v\1/g' \
		-e 's/\\o{}/\\\(\/o/g' \
		-e 's/\\O{}/\\\(\/O/g' \
		-e 's/\\EUR/\\\(eu/g' \
		-e 's/\\EUR{}/\\\(eu/g' \
		-e 's/\\euro/\\\(eu/g' \
		-e 's/\\euro{}/\\\(eu/g' \
		-e 's/---/\\\(em/g' \
		-e 's/\\textemdash/\\\(em/g' \
		-e 's/--/\\\(en/g' \
		-e 's/\\textendash/\\\(en/g' \
		-e 's/\\textcopyright/\\\(co/g' \
		-e 's/\\textregistered/\\\(rg/g' \
		-e 's/\\texttrademark/\\\(tm/g' \
		-e 's/\\textsection/\\\(sc/g' \
		-e 's/\\textexclamdown/\\\(r!/g' \
		-e 's/\\textquestiondown/\\\(r?/g' \
		-e 's/\\textquotedblleft/\\\(lq/g' \
		-e 's/\\textquotedblright/\\\(rq/g' \
		-e 's/\\quotedblbase/\\\(Bq/g' \
		-e 's/\\guillemetleft/\\\(Fo/g' \
		-e 's/\\guillemetright/\\\(Fc/g' \
		-e 's/\\textonesuperior/\\\(S1/g' \
		-e 's/\\texttwosuperior/\\\(S2/g' \
		-e 's/\\textthreesuperior/\\\(S3/g' \
		-e 's/\\textsuperscript{\([^{]*\)}/\\\*{\1\\\*}/g' \
		-e 's/\\textsubscript{\([^{]*\)}/\\\*<\1\\\*>/g' \
		-e 's/`/\\\(ga/g'
}

SepAEQ() {
	sed	-e '/^%A[[:space:]]/ s/[[:space:]][Aa][Nn][Dd]'\
'[[:space:]]/\n%A /g' \
		-e '/^%E[[:space:]]/ s/[[:space:]][Aa][Nn][Dd]'\
'[[:space:]]/\n%E /g' \
		-e '/^%Q[[:space:]]/ s/[[:space:]][Aa][Nn][Dd]'\
'[[:space:]]/\n%Q /g'
}

FuncB2RS() {
	sed -e 's/^\%/\.\\\"/' -e '/^[}]*}/d' \
		\
		-e 's/^@\([Uu][Nn][Pp][Uu][Bb][Ll][Ii][Ss][Hh][Ee][Dd]'\
'\){\([^{]*\),.*/%K \1_\2\n\%R \2/'\
		-e 's/^@\(.*\){\([^{]*\),.*/%K \1_\2/' \
		\
		-e 's/^.[Aa][Uu][Tt][Hh][Oo][Rr]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%A \1/' \
		-e 's/^.[Bb][Oo][Oo][Kk][Tt][Ti][Tt][Ll][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%B \1/' \
		-e 's/^.[Aa][Dd][Dd][Rr][Ee][Ss][Ss]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%C \1/' \
		-e 's/^.[Yy][Ee][Aa][Rr]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%D \1/' \
		-e 's/^.[Mm][Oo][Nn][Tt][Hh]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%D \1/' \
		-e 's/^.[Ee][Dd][Ii][Tt][Oo][Rr]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%E \1/' \
		-e 's/^.[Ff][Oo][Oo][Tt][Nn][Oo][Tt][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%F \1/' \
		-e 's/^.[Dd][Oo][Ii]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%G \1/' \
		-e 's/^.[Ii][Ss][Bb][Nn]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%G \1/' \
		-e 's/^.[Ii][Ss][Ss][Nn]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%G \1/' \
		-e 's/^.[Hh][Ee][Aa][Dd][Ee][Rr]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%H \1/' \
		-e 's/^.[Pp][Uu][Bb][Ll][Ii][Ss][Hh][Ee][Rr]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%I \1/' \
		-e 's/^.[Ii][Nn][Ss][Tt][Ii][Tt][Uu][Tt][Ii][Oo][Nn]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%I \1/' \
		-e 's/^.[Oo][Rr][Gg][Aa][Nn][Ii][Ss][Aa][Tt][Ii][Oo]'\
'[Nn][[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%I \1/' \
		-e 's/^.[Ss][Cc][Hh][Oo][Ll]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%I \1/' \
		-e 's/^.[Jj][Oo][Uu][Rr][Nn][Aa][Ll]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%J \1/' \
		-e 's/^.[Ss][Hh][Oo][Rr][Tt][Tt][Ii][Tt][Ll][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%L \1/' \
		-e 's/^.[Mm][Ee][Mm][Oo][Rr][Aa][Nn][Dd][Uu][Mm]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%M \1/' \
		-e 's/^.[Nn][Uu][Mm][Bb][Ee][Rr]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%N \1/' \
		-e 's/^.[Cc][Hh][Aa][Pp][Tt][Ee][Rr]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%N \1/' \
		-e 's/^.[Ee][Dd][Ii][Tt][Ii][Oo][Nn]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%N \1/' \
		-e 's/^.[Uu][Rr][Ll]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%O \1/' \
		-e 's/^.[Nn][Oo][Tt][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%O \1/' \
		-e 's/^.[Tt][Yy][Pp][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%O \1/' \
		-e 's/^.[Pp][Aa][Gg][Ee][Ss]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%P \1/' \
		-e 's/^.[Uu][Nn][Rr][Ee][Vv][Ee][Rr][Ss][Ee][Dd]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%Q \1/' \
		-e 's/^.[Hh][Oo][Ww][Pp][Uu][Bb][Ll][Ii][Ss][Hh][Ee]'\
'[Dd][[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%O \1/' \
		-e 's/^.[Ss][Ee][Rr][Ii][Ee][Ss]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%S \1/' \
		-e 's/^.[Tt][Ii][Tt][Ll][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%T \1/' \
		-e 's/^.[Ee][Mm][Aa][Ii][Ll]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%U \1/' \
		-e 's/^.[Vv][Oo][Ll][Uu][Mm][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%V \1/' \
		-e 's/^.[Ww][Hh][Ee][Rr][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%W \1/' \
		-e 's/^.[Aa][Bb][Ss][Tt][Rr][Aa][Cc][Tt]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%X \1/' \
		-e 's/^.[Aa][Nn][Nn][Oo][Tt][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%X \1/' \
		-e 's/^.[Kk][Ee][Yy][Ww][Oo][Rr][Dd][Ss]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%X \1/' \
		-e 's/^.[Tt][Oo][Cc]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%Y \1/' \
		-e 's/^.[Rr][Ee][Ff][Ee][Rr][Ee][Nn][Cc][Ee][Ss]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%Z \1/' \
		-e 's/^.[Ll][Aa][Nn][Gg][Uu][Aa][Gg][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%l \1/' \
		-e 's/^.[Pp][Rr][Ii][Cc][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%\$ \1/' \
		-e 's/^.[Cc][Oo][Pp][Yy][Rr][Ii][Gg][Hh][Tt]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%\* \1/' \
		-e 's/^.[Pp][Aa][Rr][Tt][ss]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)[}"].*/%\^ \1/'
}

FuncB2RL() {
	sed	-e 's/^.[Aa][Uu][Tt][Hh][Oo][Rr]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%A \1/' \
		-e 's/^.[Bb][Oo][Oo][Kk][Tt][Ti][Tt][Ll][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%B \1/' \
		-e 's/^.[Aa][Dd][Dd][Rr][Ee][Ss][Ss]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%C \1/' \
		-e 's/^.[Yy][Ee][Aa][Rr]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%D \1/' \
		-e 's/^.[Mm][Oo][Nn][Tt][Hh]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%D \1/' \
		-e 's/^.[Ee][Dd][Ii][Tt][Oo][Rr]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%E \1/' \
		-e 's/^.[Ff][Oo][Oo][Tt][Nn][Oo][Tt][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%F \1/' \
		-e 's/^.[Dd][Oo][Ii]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%G \1/' \
		-e 's/^.[Ii][Ss][Bb][Nn]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%G \1/' \
		-e 's/^.[Ii][Ss][Ss][Nn]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%G \1/' \
		-e 's/^.[Hh][Ee][Aa][Dd][Ee][Rr]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%H \1/' \
		-e 's/^.[Pp][Uu][Bb][Ll][Ii][Ss][Hh][Ee][Rr]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%I \1/' \
		-e 's/^.[Ii][Nn][Ss][Tt][Ii][Tt][Uu][Tt][Ii][Oo][Nn]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%I \1/' \
		-e 's/^.[Oo][Rr][Gg][Aa][Nn][Ii][Ss][Aa][Tt][Ii][Oo]'\
'[Nn][[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%I \1/' \
		-e 's/^.[Ss][Cc][Hh][Oo][Ll]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%I \1/' \
		-e 's/^.[Jj][Oo][Uu][Rr][Nn][Aa][Ll]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%J \1/' \
		-e 's/^.[Ss][Hh][Oo][Rr][Tt][Tt][Ii][Tt][Ll][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%L \1/' \
		-e 's/^.[Mm][Ee][Mm][Oo][Rr][Aa][Nn][Dd][Uu][Mm]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%M \1/' \
		-e 's/^.[Nn][Uu][Mm][Bb][Ee][Rr]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%N \1/' \
		-e 's/^.[Cc][Hh][Aa][Pp][Tt][Ee][Rr]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%N \1/' \
		-e 's/^.[Ee][Dd][Ii][Tt][Ii][Oo][Nn]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%N \1/' \
		-e 's/^.[Uu][Rr][Ll]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%O \1/' \
		-e 's/^.[Nn][Oo][Tt][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%O \1/' \
		-e 's/^.[Tt][Yy][Pp][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%O \1/' \
		-e 's/^.[Pp][Aa][Gg][Ee][Ss]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%P \1/' \
		-e 's/^.[Uu][Nn][Rr][Ee][Vv][Ee][Rr][Ss][Ee][Dd]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%Q \1/' \
		-e 's/^.[Hh][Oo][Ww][Pp][Uu][Bb][Ll][Ii][Ss][Hh][Ee]'\
'[Dd][[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%O \1/' \
		-e 's/^.[Ss][Ee][Rr][Ii][Ee][Ss]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%S \1/' \
		-e 's/^.[Tt][Ii][Tt][Ll][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%T \1/' \
		-e 's/^.[Ee][Mm][Aa][Ii][Ll]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%U \1/' \
		-e 's/^.[Vv][Oo][Ll][Uu][Mm][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%V \1/' \
		-e 's/^.[Ww][Hh][Ee][Rr][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%W \1/' \
		-e 's/^.[Aa][Bb][Ss][Tt][Rr][Aa][Cc][Tt]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%X \1/' \
		-e 's/^.[Aa][Nn][Nn][Oo][Tt][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%X \1/' \
		-e 's/^.[Kk][Ee][Yy][Ww][Oo][Rr][Dd][Ss]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%X \1/' \
		-e 's/^.[Tt][Oo][Cc]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%Y \1/' \
		-e 's/^.[Rr][Ee][Ff][Ee][Rr][Ee][Nn][Cc][Ee][Ss]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%Z \1/' \
		-e 's/^.[Ll][Aa][Nn][Gg][Uu][Aa][Gg][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%l \1/' \
		-e 's/^.[Pp][Rr][Ii][Cc][Ee]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%\$ \1/' \
		-e 's/^.[Cc][Oo][Pp][Yy][Rr][Ii][Gg][Hh][Tt]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%\* \1/' \
		-e 's/^.[Pp][Aa][Rr][Tt][ss]'\
'[[:space:]]*=[[:space:]]*[{"]\([^"]*\)*/%\^ \1/' \
	| SepAEQ \
	| sed 	-e 's/^[[:space:]]*[[:space:]]'\
'\([[:print:]]*\)[}"].*/\1/' \
		-e 's/^[[:space:]]*[[:space:]]//' \
		-e 's/[[:space:]]*$//'
}

FuncB2R() {
	FuncB2RS | FuncB2RL
}

if [ "$#" -gt '4' ] ; then echo "$EA4" >&2 ; FuncHelpShort >&2 ; exit 1
else
	case $1 in
		-h*|--help*|-\?*|\?)
			if [ "$#" -gt '1' ] ; then
				echo "$EAn" >&2 ; FuncHelpShort >&2
				exit 1
			else
				FuncHelpShort ; exit
			fi
			;;
		-v*|--version*)
			if [ "$#" -gt '1' ] ; then
				echo "$EAn" >&2 ; FuncHelpShort >&2
				exit 1
			else
				FuncVersion ; exit
			fi
			;;
		-p*|--print*)
			if [ "$#" -gt '1' ] ; then
				echo "$EAn" >&2 ; FuncHelpShort >&2
				exit 1
			else
				FI=1 ; FuncIntro ; exit
			fi
			;;
		*)
			while getopts iu OPT ; do
				case $OPT in
					i)	FI=1 ;;
					u)	FU=1 ;;
					*)	FuncHelpShort >&2
						exit 1 ;;
				esac
			done
			shift $((OPTIND - 1))
			;;
	esac
fi
if [ "$#" -gt '2' ] ; then echo "$EA2" >&2 ; FuncHelpShort >&2 ; exit 1
elif [ -r "$1" -a "$2" ] ; then
	case $2 in
		-|'')	FuncIntro ; FuncUTF82R < "$1" \
			| FuncB2R | FuncChB2R ;;
		*)
			if [ "$1" != "$2" ] ; then
					FuncIntro > "$2"
					FuncUTF82R < "$1" | FuncB2R \
					| FuncChB2R >> "$2"
				else
					echo "$EI" >&2
					FuncHelpShort >&2
					exit 1
			fi
			;;
	esac
elif [ -r "$1" ] ; then
	FuncIntro ; FuncUTF82R < "$1"| FuncB2R | FuncChB2R 
	else
	case $1 in
		-|'')
		case $2 in
			-|'')	FuncIntro ; FuncUTF82R | FuncB2R \
				| FuncChB2R ;;
			*)	FuncIntro > "$2" ; FuncUTF82R \
				| FuncB2R | FuncChB2R >> "$2" ;;
		esac
		;;
		-*|--*)	echo "$EO" >&2 ; FuncHelpShort >&2 ; exit 1 ;;
		*)	echo "$EF" >&2 ; FuncHelpShort >&2 ; exit 1 ;;
	esac
fi
